﻿using System;
using System.Security.Cryptography;
using System.Text;
using OpenSSL.Core;
using RSA = OpenSSL.Crypto.RSA;

namespace eCommSignature
{
    public static class Signature
    {
        /// <summary>
        /// Generate RSA signature
        /// </summary>
        /// <param name="macData">MAC data</param>
        /// <param name="prefix">Prefix</param>
        /// <param name="privateKeyFileName">PEM private key file name</param>
        /// <exception cref="ArgumentException">
        /// </exception>
        /// <returns>Signature as hexadecimal string</returns>
        public static string CreateSignature(string macData, string prefix, string privateKeyFileName)
        {
            //all parameters must be provided
            if (string.IsNullOrEmpty(macData)) 
                throw new ArgumentException("MAC data should be provided");
            if(string.IsNullOrEmpty(prefix))
                throw new ArgumentException("Prefix row should be provided");
            if( string.IsNullOrEmpty(privateKeyFileName))
                throw new ArgumentException("Private key PEM file name should be provided");

            //Instantiate OpenSSL.Crypto.RSA object
            RSA rsa = RSA.FromPrivateKey(BIO.File(privateKeyFileName, "r"));

            //Build data row for signing
            //Prefix and MD5 hash of MAC data will be padded in order to obtain 
            //a data of length equal to RSA private key length
            //First 2 bytes of padding will be 00 01 and the rest - FF
            StringBuilder data = new StringBuilder();

            //Compute MD5 hash (16 bytes) of MAC data
            string md5Data = CalculateMD5Hash(macData);

            const string first = "0001";
            
            //Append first two bytes
            data.Append(first);

            //Compute padding length (in bytes)
            int paddingLength = rsa.Size - md5Data.Length / 2 - prefix.Length /2 - first.Length / 2;
            //Append padding bytes
            for (int i = 0; i < paddingLength; i++)
                data.Append("FF");
            //Append prefix
            data.Append(prefix);
            //Append MD5 hash of MAC data
            data.Append(md5Data);

            //Get bytes
            byte[] dataBytes = HexToByteArray(data.ToString());

            //RSA Sign obtained data
            //Use no padding, as the length of the message is equal to the length of the key
            byte[] encrypted = rsa.PrivateEncrypt(dataBytes, RSA.Padding.None);

            //Return result as hexadecimal string
            return ByteArrayToHex(encrypted).ToUpperInvariant();
        }

        /// <summary>
        /// Verify p_sign field
        /// </summary>
        /// <param name="pSign">Value of encrypted p_sign field</param>
        /// <param name="modulus">Hexadecimal  representation of public RSA key's modulus</param>
        /// <param name="exponent">Hexadecimal  representation of public RSA key's exponent</param>
        /// <exception cref="ArgumentException">
        /// </exception>
        /// <returns>Hexadecimal representation of decrypted p_sign field</returns>
        public static string VerifySignature(string pSign, string modulus, string exponent)
        {
            //all parameters must be provided
            if (string.IsNullOrEmpty(pSign))
                throw new ArgumentException("Value of encrypted p_sign field must be provided");
            if (string.IsNullOrEmpty(exponent))
                throw new ArgumentException("Public key's exponent must be provided");
            if (string.IsNullOrEmpty(modulus))
                throw new ArgumentException("Public key's modulus must be provided");
            if (pSign.Length != modulus.Length)
                throw new ArgumentException("p_sign's and modulus's lengths must be equal");

            //Create RSA key from exponent and modulus
            RSA rsa = new RSA();
            rsa.PublicExponent = BigNumber.FromHexString(exponent);
            rsa.PublicModulus = BigNumber.FromHexString(modulus);

            //Decrypt (verify) signature
            byte[] decrypted = rsa.PublicDecrypt(HexToByteArray(pSign), RSA.Padding.None);



            //Return hexadecimal representation of the result
            string hexaDecrypted = ByteArrayToHex(decrypted);
            return hexaDecrypted;
        }

        #region Private methods

        private static string CalculateMD5Hash(string input)
        {
            //calculate MD5 hash from input
            MD5 md5 = MD5.Create();
            byte[] inputBytes = Encoding.ASCII.GetBytes(input);
            byte[] hash = md5.ComputeHash(inputBytes);

            //Return hexa-decimal representation of the obtained hash
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < hash.Length; i++)
            {
                sb.Append(hash[i].ToString("X2"));
            }
            return sb.ToString();
        }

        private static byte[] HexToByteArray(string hexa)
        {
            byte[] array = new byte[hexa.Length / 2];
            for (int i = 0; i < hexa.Length; i += 2)
            {
                array[i / 2] = Convert.ToByte(hexa.Substring(i, 2), 16);
            }
            return array;
        }

        private static string ByteArrayToHex(byte[] array)
        {
            StringBuilder sb = new StringBuilder(array.Length*2);
            foreach (byte b in array)
            {
                sb.AppendFormat("{0:x2}", b);
            }
            return sb.ToString();
        }

        #endregion
    }
}
